function [valic,valic_corr,nlag]=lagselec_var(y,pmax,lagsel,ic,fd,const,steps,rounds)
% PURPOSE: determine lag length or lag structure in VAR-model via
%          - information criterion:  lagselec='infc'
%          - heuristic optimization: lagselec='heur'
%
% where:    y          =    a (nobs x 1) vector
%           pmax       =    maximum lag length
%           lagsel     =    'infc': selection by information criterion
%                           'heur': selection by heuristc optimization
%                           (threshold accepting)
%                      =    'gts': general to specific approach
%           ic         =    information criterion: AIC, BIC or HQ
%           fd         =     1 first difference, 0=levels
%
%           optional arguments:
%           steps      =    number of steps heur TA 
%           rounds     =    number of rounds heur TA
%
%          Note: Uses functions of LeSage-Toolbox
%          Note: Only works with Constant because of vare.m
%          Note: valic returns information criteria based on adjusted
%                sample (T-pmax) valic_corr retruns information criteria 
%                on correct sample (T-p)
%
% ---------------------------------------------------
% RETURNS: 
%         - valic:      value of information criterion
%         - valic_corr: value of information criterion based on correct
%                       sample length
%         - lags:       lag length or lag structure
%
% written by 
% Frauke Schleer
% schleer@zew.de


if fd==1
    pmax=pmax-1;
end

[~, rows]=size(y);

switch lagsel
     case 'infc' %coincide with EViews
       
         % initialise matrices
         ll=zeros(pmax,1);
         ll_corr=zeros(pmax,1);
         icr=zeros(pmax,1);
         valic_corr=zeros(pmax,1);
         
         % determine lag length based on information criteria
         for p=1:pmax
             if const==0
                results_vare = vare_wc(y,p);
                results_vare_pmax = vare_pmax_wc(y,p,pmax); 
             else
                results_vare = vare(y,p);
                results_vare_pmax = vare_pmax(y,p,pmax);
             end
             nvar=results_vare.nvar;
             nobsc=results_vare.nobs; %adjusted lags
             nobs= results_vare_pmax.nobse; 
             ll(p,1)=-nobs/2*(rows*(1+log(2*pi))+log(results_vare_pmax.detsig));
             ll_corr(p,1)=-nobsc/2*(rows*(1+log(2*pi))+log(results_vare(1,1).detsig));
             icr(p,1)=infcr(ic,ll(p,1),nobs,nvar*rows,'multi');
             valic_corr(p,1)=infcr(ic,ll_corr(p,1),nobsc,nvar*rows,'multi');
         end

         [valic,rr]=min(icr);
         nlag=rr;
         valic_corr=valic_corr(rr,1);
         
  
    case 'heur'
       
        % determine lag length based on heuristic optimitzation
        results_varho = vare_ho(y,pmax,steps,rounds,ic,const);
        nlag=results_varho.lagstructure;
        valic=results_varho.valinf;
        valic_corr=results_varho.valinf_corr;        
        
        lagstr_pre=reshape(nlag,pmax,[]);
        lagstr_pre=lagstr_pre.*repmat((1:pmax)',[1,size(lagstr_pre,2)]);
        nlag_max=max(max(lagstr_pre));
       
        if nlag_max==pmax;
           warning('optimized lag length is equal to initially chosen maximum lag length');
        end

        
end

